/**
 * SIMPLE SITEMAP GENERATOR
 * =========================
 * Generates a single comprehensive sitemap.xml
 */

const fs = require('fs');
const path = require('path');

// Load data
const services = require('../data/services.json');
const cities = require('../data/cities.json');
const blogArticles = require('../data/blog-articles.json');

const FULL_DOMAIN = 'https://rapidserviceconnect.com';

function generateSitemap(OUTPUT_DIR) {
  console.log('Building sitemap.xml...');

  const today = new Date().toISOString().split('T')[0];

  // Helper to add URL
  function addUrl(path, priority = '0.5', changefreq = 'weekly') {
    const urlPath = path === '/' ? '' : (path.startsWith('/') ? path : `/${path}`);
    let finalUrl = `${FULL_DOMAIN}${urlPath}`;

    if (urlPath !== '' && !urlPath.endsWith('/') && !urlPath.endsWith('.html') && !urlPath.endsWith('.xml')) {
      finalUrl += '/';
    }

    return `
  <url>
    <loc>${finalUrl}</loc>
    <lastmod>${today}</lastmod>
    <changefreq>${changefreq}</changefreq>
    <priority>${priority}</priority>
  </url>`;
  }

  const allUrls = [];

  // Static Pages
  allUrls.push('\n  <!-- MAIN PAGES -->');
  allUrls.push(addUrl('/', '1.0', 'daily'));
  allUrls.push(addUrl('/about', '0.8', 'monthly'));
  allUrls.push(addUrl('/contact', '0.8', 'monthly'));
  allUrls.push(addUrl('/how-it-works', '0.8', 'monthly'));
  allUrls.push(addUrl('/blog', '0.9', 'daily'));

  // Services
  allUrls.push('\n  <!-- SERVICES -->');
  services.forEach(service => {
    allUrls.push(addUrl(`/${service.slug}`, '0.9', 'weekly'));
  });

  // Cities
  allUrls.push('\n  <!-- CITIES -->');
  cities.forEach(city => {
    allUrls.push(addUrl(`/${city.slug}`, '0.9', 'weekly'));
  });

  // Money Pages (Service + City combinations)
  allUrls.push('\n  <!-- SERVICE + CITY PAGES -->');
  services.forEach(service => {
    cities.forEach(city => {
      allUrls.push(addUrl(`/${service.slug}-${city.slug}`, '0.8', 'weekly'));
    });
  });

  // Blog Articles
  allUrls.push('\n  <!-- BLOG ARTICLES -->');
  blogArticles.forEach(article => {
    allUrls.push(addUrl(`/blog/${article.slug}`, '0.7', 'monthly'));
  });

  // Legal Pages
  allUrls.push('\n  <!-- LEGAL PAGES -->');
  allUrls.push(addUrl('/privacy-policy', '0.5', 'yearly'));
  allUrls.push(addUrl('/terms', '0.5', 'yearly'));
  allUrls.push(addUrl('/disclaimer', '0.5', 'yearly'));

  const sitemapContent = `<?xml version="1.0" encoding="UTF-8"?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
${allUrls.join('')}
</urlset>`;

  const sitemapPath = path.join(OUTPUT_DIR, 'sitemap.xml');
  fs.writeFileSync(sitemapPath, sitemapContent);

  const totalUrls = 8 + services.length + cities.length + (services.length * cities.length) + blogArticles.length;

  console.log(`\n✅ SITEMAP GENERATED`);
  console.log(`===================`);
  console.log(`📋 sitemap.xml: ${totalUrls} URLs`);
  console.log(`   ├─ Main Pages: 8 URLs`);
  console.log(`   ├─ Services: ${services.length} URLs`);
  console.log(`   ├─ Cities: ${cities.length} URLs`);
  console.log(`   ├─ Service+City Pages: ${services.length * cities.length} URLs`);
  console.log(`   └─ Blog Articles: ${blogArticles.length} URLs`);
  console.log(`\n🎯 All URLs organized with category comments`);
}

module.exports = { generateSitemap };
