/**
 * CITY MONEY PAGE GENERATOR - ENTERPRISE SEO OPTIMIZED
 * =====================================================
 * 
 * These pages are revenue drivers - optimized for:
 * - Local SEO (Google Maps, Local Pack)
 * - Emergency service keywords
 * - High commercial intent
 * - Fast indexing
 * - LLM answer engines
 * - MarketCall conversion
 */

const fs = require('fs');
const path = require('path');
const {
  generateFAQSchema,
  generateHowToSchema,
  generateLocalBusinessSchema,
  generateBreadcrumbSchema,
  getHeader,
  getNavigation,
  getFooter,
  getDisclaimer,
  services,
  cities,
  blogArticles
} = require('./seo-utils');

/**
 * Generate all city money pages (SERVICE x CITY matrix)
 */
function generateCityMoneyPages(OUTPUT_DIR, ensureDir) {
  let count = 0;
  const totalPages = services.length * cities.length;

  console.log(`Generating ${totalPages} city money pages...`);

  services.forEach((service, sIdx) => {
    cities.forEach((city, cIdx) => {
      const trackingNumber = `[[TRACKING_NUMBER_${service.id.toUpperCase()}_${city.id.toUpperCase()}]]`;

      // Get related blog articles for this service
      const relatedBlogs = blogArticles
        .filter(a => a.service === service.id)
        .slice(0, 3);

      // Get 2 other services in same city for cross-linking
      const otherServices = services
        .filter(s => s.id !== service.id)
        .slice(0, 2);

      // Generate entity-focused FAQs
      const faqItems = [
        {
          question: `Is ${service.name.toLowerCase()} available 24/7 in ${city.name}?`,
          answer: `Yes, we connect you with ${service.name.toLowerCase()} professionals in ${city.name} who offer 24-hour emergency response. Service availability may vary, but we prioritize connecting you with providers who can help immediately.`
        },
        {
          question: `How quickly can I get ${service.name.toLowerCase()} help in ${city.name}, ${city.state}?`,
          answer: `Most ${city.name} customers are connected with a ${service.name.toLowerCase()} professional within minutes. Response times depend on your specific location within ${city.name} and current provider availability.`
        },
        {
          question: `Are the ${service.name.toLowerCase()} providers in ${city.name} licensed and insured?`,
          answer: `Yes, all ${service.name.toLowerCase()} providers in our ${city.name} network are required to maintain proper licensing and insurance for ${city.state}. We verify credentials regularly.`
        },
        {
          question: `What types of ${service.name.toLowerCase()} emergencies do you handle in ${city.name}?`,
          answer: `We connect ${city.name} residents with professionals who handle all types of ${service.name.toLowerCase()} emergencies including ${service.emergencies.slice(0, 3).map(e => e.toLowerCase()).join(', ')}, and more.`
        },
        {
          question: `Is there a cost to use your ${service.name.toLowerCase()} referral service in ${city.name}?`,
          answer: `No, our referral service is completely free. You only pay the ${service.name.toLowerCase()} provider directly for their work. There are no hidden fees from Rapid Service Connect.`
        },
        {
          question: `What areas of ${city.name} do you serve?`,
          answer: `We serve all areas of ${city.name} and surrounding ${city.state} communities. Our network covers the entire ${city.region} region.`
        }
      ];

      // Breadcrumbs for navigation
      const breadcrumbs = [
        { name: "Home", url: "/" },
        { name: "Services", url: "/services/" },
        { name: service.name, url: `/services/${service.slug}/` },
        { name: `${city.name}, ${city.state}`, url: `/services/${service.slug}/${city.slug}/` }
      ];

      const html = `${getHeader()}
  <title>Emergency ${service.name} in ${city.name}, ${city.state} – 24/7 Fast Response | Call Now</title>
  <meta name="description" content="Need ${service.name.toLowerCase()} in ${city.name}? Connect with verified 24/7 professionals instantly. Serving all of ${city.name}, ${city.state}. Call now for immediate help.">
  <meta property="og:title" content="Emergency ${service.name} in ${city.name}, ${city.state} – 24/7">
  <meta property="og:description" content="${service.name} professionals in ${city.name}. Call for instant connection to local experts.">
  <meta property="og:type" content="website">
  <meta property="og:url" content="https://www.rapidserviceconnect.com/services/${service.slug}/${city.slug}/">
  <link rel="canonical" href="https://www.rapidserviceconnect.com/services/${service.slug}/${city.slug}/">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
  <link rel="stylesheet" href="/css/mega-menu.css">
  <link rel="stylesheet" href="/css/chatbot.css">
  ${generateFAQSchema(faqItems)}
  ${generateLocalBusinessSchema(service.name, city.name, city.stateFullName)}
  ${generateBreadcrumbSchema(breadcrumbs)}
</head>
<body>
  ${getNavigation(services, cities)}
  
  <main id="main-content">
    <nav class="breadcrumbs" aria-label="Breadcrumb">
      <ol>
        ${breadcrumbs.map((crumb, idx) => `<li${idx === breadcrumbs.length - 1 ? ' aria-current="page"' : ''}><a href="${crumb.url}">${crumb.name}</a></li>`).join('')}
      </ol>
    </nav>
    
    <div class="page-header">
      <div class="container">
        <h1>Emergency ${service.name} in ${city.name}, ${city.state} – 24/7 Fast Response</h1>
        <p>
          Facing a ${service.name.toLowerCase()} emergency in ${city.name}? Get connected immediately with qualified, licensed 
          professionals who serve ${city.name} and the surrounding ${city.region} area. Available 24 hours a day, 7 days a week.
        </p>
      </div>
    </div>
    
    <section class="section cta-section">
      <div class="container">
        <div class="cta-block">
          <h2>Call Now for Immediate ${service.name} Help in ${city.name}</h2>
          <a href="tel:${trackingNumber}" class="cta-phone" aria-label="Call for emergency ${service.name.toLowerCase()} service">${trackingNumber}</a>
          <p>
            <strong>Available 24/7</strong> • Local ${city.name} Professionals • 
            <strong>Licensed & Insured</strong> • Fast Emergency Response
          </p>
        </div>
      </div>
    </section>
    
    <section class="content-section">
      <div class="container">
        <article class="content-wrapper">
          <h2>What Types of ${service.name} Emergencies We Handle in ${city.name}</h2>
          <p>
            Whether you're a homeowner or business in ${city.name}, ${city.state}, ${service.name.toLowerCase()} emergencies 
            can happen at any time. That's why Rapid Service Connect provides instant connections to ${city.name} 
            professionals who specialize in handling urgent situations.
          </p>
          
          <h3>Common ${service.name} Emergencies in ${city.name}:</h3>
          <ul>
            ${service.emergencies.map(emergency => `<li><strong>${emergency}</strong> – Get immediate professional help in ${city.name}</li>`).join('\n            ')}
          </ul>
          
          <h2>Why Choose Rapid Service Connect for ${service.name} in ${city.name}?</h2>
          <p>
            When you're facing a ${service.name.toLowerCase()} emergency in ${city.name}, you need help fast. Here's why 
            ${city.name} residents trust Rapid Service Connect:
          </p>
          
          <ul>
            <li><strong>24/7 Availability:</strong> ${service.name} emergencies don't wait for business hours, and neither do we. Call anytime, day or night.</li>
            <li><strong>Local ${city.name} Professionals:</strong> We connect you with providers who know ${city.name} and the ${city.region} area.</li>
            <li><strong>Pre-Verified Network:</strong> All ${service.name.toLowerCase()} providers are licensed and insured in ${city.state}.</li>
            <li><strong>Fast Response:</strong> Most ${city.name} callers are connected with a professional within minutes.</li>
            <li><strong>Free Referral Service:</strong> Our service is completely free—you only pay the provider for their work.</li>
          </ul>
          
          <h2>What to Do Before ${service.name} Professionals Arrive in ${city.name}</h2>
          <p>
            While waiting for your ${service.name.toLowerCase()} professional to arrive in ${city.name}, here are some 
            important safety tips:
          </p>
          
          <ul>
            ${service.tips.map(tip => `<li>${tip}</li>`).join('\n            ')}
          </ul>
          
          <h2>What to Expect When You Call for ${service.name} in ${city.name}</h2>
          <p>
            When you call Rapid Service Connect for ${service.name.toLowerCase()} help in ${city.name}, here's what happens:
          </p>
          
          <ol>
            <li><strong>Immediate Connection:</strong> You'll be connected with a ${service.name.toLowerCase()} professional serving ${city.name}.</li>
            <li><strong>Direct Communication:</strong> Speak directly with the provider about your specific emergency situation.</li>
            <li><strong>Quick Response:</strong> The provider will give you an estimated arrival time for ${city.name}.</li>
            <li><strong>Professional Service:</strong> Licensed, insured ${service.name.toLowerCase()} experts handle your emergency.</li>
            <li><strong>Direct Payment:</strong> You pay the provider directly for their work—no fees from us.</li>
          </ol>
          
          <h2>Serving All Areas of ${city.name}, ${city.state}</h2>
          <p>
            Our ${service.name.toLowerCase()} network covers the entire ${city.name} metropolitan area, including all 
            neighborhoods and surrounding communities in ${city.region}. With approximately ${city.population} people calling 
            ${city.name} home, we understand the importance of fast, reliable ${service.name.toLowerCase()} response throughout 
            the city.
          </p>
          
          <h2>Other Emergency Services We Offer in ${city.name}</h2>
          <p>
            In addition to ${service.name.toLowerCase()}, Rapid Service Connect also connects ${city.name} residents with 
            professionals for these emergency services:
          </p>
          
          <ul>
            ${otherServices.map(s => `<li><a href="/services/${s.slug}/${city.slug}/">${s.name} in ${city.name}</a> – ${s.description}</li>`).join('\n            ')}
            <li><a href="/cities/${city.slug}/">View all ${services.length} emergency services available in ${city.name}</a></li>
          </ul>
        </article>
      </div>
    </section>
    
    <section class="section" style="background: var(--color-gray-100);">
      <div class="container">
        <div class="cta-block">
          <h2>Don't Wait – Get ${service.name} Help in ${city.name} Now</h2>
          <p>Emergency situations require immediate action. Call now to connect with a local ${service.name.toLowerCase()} professional.</p>
          <a href="tel:${trackingNumber}" class="btn btn-primary btn-large">Call ${trackingNumber}</a>
        </div>
      </div>
    </section>
    
    <section class="content-section faq-section">
      <div class="container">
        <div class="content-wrapper">
          <h2>${service.name} in ${city.name} – Frequently Asked Questions</h2>
          
          ${faqItems.map(faq => `<div class="faq-item">
            <button class="faq-question">${faq.question}</button>
            <div class="faq-answer">
              <p>${faq.answer}</p>
            </div>
          </div>`).join('\n          ')}
        </div>
      </div>
    </section>
    
    <section class="section" style="background: var(--color-gray-100);">
      <div class="container">
        <div class="content-wrapper">
          <h2>Learn More About ${service.name}</h2>
          <p>Get expert advice and tips from our resource library:</p>
          <ul>
            ${relatedBlogs.map(blog => `<li><a href="/blog/${blog.slug}/">${blog.title}</a></li>`).join('\n            ')}
            <li><a href="/services/${service.slug}/">Complete ${service.name} resource center</a></li>
          </ul>
        </div>
      </div>
    </section>
    
    <section class="content-section">
      <div class="container">
        <div class="content-wrapper">
          ${getDisclaimer()}
        </div>
      </div>
    </section>
  </main>
  
  ${getFooter(services, cities)}`;

      const outputPath = path.join(OUTPUT_DIR, 'services', service.slug, city.slug, 'index.html');
      ensureDir(path.dirname(outputPath));
      fs.writeFileSync(outputPath, html);

      count++;
      if (count % 100 === 0) {
        console.log(`  → Generated ${count}/${totalPages} city money pages...`);
      }
    });
  });

  console.log(`✓ Generated ${count} SEO-optimized city money pages`);
}

module.exports = {
  generateCityMoneyPages
};
