/**
 * SERVICE HUB + REMAINING PAGE GENERATORS
 * ========================================
 * Service hubs, city hubs, blog pages, and core pages
 */

const fs = require('fs');
const path = require('path');
const {
  generateFAQSchema,
  generateHowToSchema,
  generateBreadcrumbSchema,
  getHeader,
  getNavigation,
  getFooter,
  getDisclaimer,
  services,
  cities,
  blogArticles
} = require('./seo-utils');

/**
 * Generate Service Hub Pages (Legacy/Basic)
 * Kept for reference, but replaced by premium-service-generator.js
 */
function generateServiceHubPages(OUTPUT_DIR, ensureDir) {
  services.forEach(service => {
    const relatedBlogs = blogArticles.filter(a => a.service === service.id);

    const faqItems = [
      {
        question: `What is ${service.name.toLowerCase()}?`,
        answer: `${service.name} involves professional help for ${service.emergencies.slice(0, 2).map(e => e.toLowerCase()).join(', ')} and other urgent situations. Our network connects you with licensed professionals 24/7.`
      },
      {
        question: `Do you provide ${service.name.toLowerCase()} services directly?`,
        answer: `No, we are a referral service. We connect you with independent, licensed ${service.name.toLowerCase()} professionals in your area who provide the actual services.`
      },
      {
        question: `Is ${service.name.toLowerCase()} available 24/7?`,
        answer: `Yes, we connect you with ${service.name.toLowerCase()} professionals who offer 24-hour emergency response in cities across the United States.`
      },
      {
        question: `Which cities do you serve for ${service.name.toLowerCase()}?`,
        answer: `We currently serve ${cities.length} major U.S. cities for ${service.name.toLowerCase()} referrals. View the list below to find your city.`
      }
    ];

    const breadcrumbs = [
      { name: "Home", url: "/" },
      { name: service.name, url: `/${service.slug}/` }
    ];

    const html = `${getHeader()}
  <title>${service.name} Near You – 24/7 Emergency Connection | Rapid Service Connect</title>
  <meta name="description" content="Connect instantly with trusted 24/7 ${service.name.toLowerCase()} professionals. Serving ${cities.length} U.S. cities. Get immediate help for ${service.emergencies[0].toLowerCase()} and more.">
  <link rel="canonical" href="http://rapidserviceconnect.com//${service.slug}/">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
  <link rel="stylesheet" href="/css/mega-menu.css">
  <link rel="stylesheet" href="/css/chatbot.css">
  ${generateFAQSchema(faqItems)}
  ${generateBreadcrumbSchema(breadcrumbs)}
</head>
<body>
  ${getNavigation(services, cities)}
  
  <main id="main-content">
    <nav class="breadcrumbs" aria-label="Breadcrumb">
      <ol>
        ${breadcrumbs.map((crumb, idx) => `<li${idx === breadcrumbs.length - 1 ? ' aria-current="page"' : ''}><a href="${crumb.url}">${crumb.name}</a></li>`).join('')}
      </ol>
    </nav>
    
    <div class="page-header">
      <div class="container">
        <h1>${service.name} – 24/7 Emergency Help in ${cities.length} U.S. Cities</h1>
        <p>Connect instantly with verified ${service.name.toLowerCase()} professionals in your area. Available 24/7 for emergencies.</p>
      </div>
    </div>
    
    <section class="content-section">
      <div class="container">
        <article class="content-wrapper">
          <h2>Common ${service.name} Emergencies We Handle</h2>
          <p>Our network of ${service.name.toLowerCase()} professionals is ready to help with:</p>
          <ul>
            ${service.emergencies.map(e => `<li><strong>${e}</strong></li>`).join('\n            ')}
          </ul>
          
          <h2>Safety Tips While Waiting for ${service.name} Help</h2>
          <ul>
            ${service.tips.map(tip => `<li>${tip}</li>`).join('\n            ')}
          </ul>
        </article>
      </div>
    </section>
    
    <section class="content-section" style="background: var(--color-gray-100);">
      <div class="container">
        <div class="content-wrapper">
          <h2>Find ${service.name} in Your City</h2>
          <p>Select your city to connect with local ${service.name.toLowerCase()} professionals:</p>
          <div class="cities-grid">
            ${cities.map(c => `<a href="/${service.slug}-${c.slug}/" class="city-link">${c.name}, ${c.state}</a>`).join('\n            ')}
          </div>
        </div>
      </div>
    </section>
    
    <section class="content-section faq-section">
      <div class="container">
        <div class="content-wrapper">
          <h2>Frequently Asked Questions About ${service.name}</h2>
          ${faqItems.map(faq => `<div class="faq-item">
            <button class="faq-question">${faq.question}</button>
            <div class="faq-answer"><p>${faq.answer}</p></div>
          </div>`).join('\n          ')}
        </div>
      </div>
    </section>
    
    <section class="section" style="background: var(--color-gray-100);">
      <div class="container">
        <div class="content-wrapper">
          <h2>${service.name} Resources and Guides</h2>
          <ul>
            ${relatedBlogs.slice(0, 5).map(blog => `<li><a href="/blog/${blog.slug}/">${blog.title}</a></li>`).join('\n            ')}
          </ul>
        </div>
      </div>
    </section>
    
    <section class="content-section">
      <div class="container">
        <div class="content-wrapper">
          ${getDisclaimer()}
        </div>
      </div>
    </section>
  </main>
  
  ${getFooter(services, cities)}`;

    const outputPath = path.join(OUTPUT_DIR, service.slug, 'index.html');
    ensureDir(path.dirname(outputPath));
    fs.writeFileSync(outputPath, html);
  });

  console.log(`✓ Generated ${services.length} service hub pages (LEGACY)`);
}

/**
 * Generate City Hub Pages
 */
function generateCityHubPages(OUTPUT_DIR, ensureDir) {
  cities.forEach(city => {
    const breadcrumbs = [
      { name: "Home", url: "/" },
      { name: `${city.name}, ${city.state}`, url: `/${city.slug}/` }
    ];

    const html = `${getHeader()}
  <title>Emergency Services in ${city.name}, ${city.state} – 24/7 Local Help | Rapid Service Connect</title>
  <meta name="description" content="Connect with 24/7 emergency services in ${city.name}, ${city.state}. ${services.length} services available including plumbing, HVAC, electrical, and more.">
  <link rel="canonical" href="http://rapidserviceconnect.com//${city.slug}/">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
  <link rel="stylesheet" href="/css/mega-menu.css">
  <link rel="stylesheet" href="/css/chatbot.css">
  ${generateBreadcrumbSchema(breadcrumbs)}
</head>
<body>
  ${getNavigation(services, cities)}
  
  <main id="main-content">
    <nav class="breadcrumbs" aria-label="Breadcrumb">
      <ol>
        ${breadcrumbs.map((crumb, idx) => `<li${idx === breadcrumbs.length - 1 ? ' aria-current="page"' : ''}><a href="${crumb.url}">${crumb.name}</a></li>`).join('')}
      </ol>
    </nav>
    
    <div class="page-header">
      <div class="container">
        <h1>Emergency Services in ${city.name}, ${city.state} – 24/7 Local Help</h1>
        <p>Find immediate help from local professionals in ${city.name}. ${services.length} emergency services available 24/7.</p>
      </div>
    </div>
    
    <section class="section">
      <div class="container">
        <h2 class="text-center mb-8">Available Emergency Services in ${city.name}</h2>
        <div class="services-grid">
          ${services.map(s => `<a href="/${s.slug}-${city.slug}/" class="service-card">
            <span class="service-icon" role="img" aria-label="${s.name} icon">${s.icon}</span>
            <h3>${s.name}</h3>
            <p>24/7 ${s.name.toLowerCase()} help in ${city.name}, ${city.state}</p>
          </a>`).join('\n          ')}
        </div>
      </div>
    </section>
    
    <section class="content-section" style="background: var(--color-gray-100);">
      <div class="container">
        <div class="content-wrapper">
          <h2>About Emergency Services in ${city.name}, ${city.state}</h2>
          <p>
            ${city.name}, with a population of approximately ${city.population}, is part of the ${city.region} region. 
            Rapid Service Connect connects ${city.name} residents with verified local professionals for all types of 
            emergency services—available 24 hours a day, 7 days a week.
          </p>
          ${getDisclaimer()}
        </div>
      </div>
    </section>
  </main>
  
  ${getFooter(services, cities)}`;

    const outputPath = path.join(OUTPUT_DIR, city.slug, 'index.html');
    ensureDir(path.dirname(outputPath));
    fs.writeFileSync(outputPath, html);
  });

  console.log(`✓ Generated ${cities.length} city hub pages`);
}

/**
 * Generate Blog Pages
 */
function generateBlogPages(OUTPUT_DIR, ensureDir) {
  // Blog index
  const blogIndexHtml = `${getHeader()}
  <title>Emergency Services Resources & Guides | Rapid Service Connect</title>
  <meta name="description" content="Expert advice and guides for handling home emergencies. Resources for plumbing, electrical, HVAC, roofing, and more.">
  <link rel="canonical" href="http://rapidserviceconnect.com//blog/">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
  <link rel="stylesheet" href="/css/mega-menu.css">
  <link rel="stylesheet" href="/css/chatbot.css">
</head>
<body>
  ${getNavigation(services, cities)}
  
  <main id="main-content">
    <div class="page-header">
      <div class="container">
        <h1>Emergency Services Resources & Guides</h1>
        <p>Expert tips and guides for handling home emergencies</p>
      </div>
    </div>
    
    <section class="section">
      <div class="container">
        <div class="content-wrapper">
          ${services.map(service => {
    const serviceArticles = blogArticles.filter(a => a.service === service.id);
    return `<h3>${service.name} Resources</h3>
          <ul class="mb-8">
            ${serviceArticles.map(a => `<li><a href="/blog/${a.slug}/">${a.title}</a></li>`).join('\n            ')}
          </ul>`;
  }).join('\n')}
        </div>
      </div>
    </section>
  </main>
  
  ${getFooter(services, cities)}`;

  ensureDir(path.join(OUTPUT_DIR, 'blog'));
  fs.writeFileSync(path.join(OUTPUT_DIR, 'blog', 'index.html'), blogIndexHtml);

  // Individual articles
  let count = 0;
  blogArticles.forEach(article => {
    const service = services.find(s => s.id === article.service);
    const relatedCities = cities.slice(0, 3);

    const breadcrumbs = [
      { name: "Home", url: "/" },
      { name: "Resources", url: "/blog/" },
      { name: article.title, url: `/blog/${article.slug}/` }
    ];

    const html = `${getHeader()}
  <title>${article.title}</title>
  <meta name="description" content="${article.metaDescription}">
  <link rel="canonical" href="http://rapidserviceconnect.com//blog/${article.slug}/">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
  <link rel="stylesheet" href="/css/mega-menu.css">
  <link rel="stylesheet" href="/css/chatbot.css">
  ${generateBreadcrumbSchema(breadcrumbs)}
</head>
<body>
  ${getNavigation(services, cities)}
  
  <main id="main-content">
    <article class="blog-article">
      <div class="container">
        <nav class="breadcrumbs" aria-label="Breadcrumb">
          <ol>
            ${breadcrumbs.map((crumb, idx) => `<li${idx === breadcrumbs.length - 1 ? ' aria-current="page"' : ''}><a href="${crumb.url}">${crumb.name}</a></li>`).join('')}
          </ol>
        </nav>
        
        <h1>${article.title}</h1>
        <div class="article-meta">Category: <a href="/${service.slug}/">${service.name}</a></div>
        
        <div class="article-content">
          <h2>Understanding ${service.name} Situations</h2>
          <p>When facing ${service.name.toLowerCase()} emergencies, knowing what to do can make all the difference in protecting your property and ensuring safety.</p>
          
          <h2>What You Need to Know</h2>
          <p>${article.metaDescription}</p>
          
          <h2>When to Call for Professional ${service.name}</h2>
          <p>Contact a professional immediately if you experience:</p>
          <ul>
            ${service.emergencies.map(e => `<li>${e}</li>`).join('\n            ')}
          </ul>
          
          <h2>Get Help in Your City</h2>
          <p>Need immediate ${service.name.toLowerCase()} help? We connect you with professionals in major cities:</p>
          <ul>
            ${relatedCities.map(c => `<li><a href="/${service.slug}-${c.slug}/">${service.name} in ${c.name}, ${c.state}</a></li>`).join('\n            ')}
            <li><a href="/${service.slug}/">View all ${cities.length} cities served</a></li>
          </ul>
        </div>
        
        <div class="cta-block mt-12">
          <h2>Need Immediate ${service.name} Help?</h2>
          <p>Connect with a 24/7 professional in your city right now.</p>
          <a href="/${service.slug}/" class="btn btn-primary btn-large">Get Help Now</a>
        </div>
        
        <div class="mt-12">${getDisclaimer()}</div>
      </div>
    </article>
  </main>
  
  ${getFooter(services, cities)}`;

    ensureDir(path.join(OUTPUT_DIR, 'blog', article.slug));
    fs.writeFileSync(path.join(OUTPUT_DIR, 'blog', article.slug, 'index.html'), html);
    count++;
  });

  console.log(`✓ Generated ${count + 1} blog pages (including index)`);
}

/**
 * Generate Core/Legal Pages
 */
function generateCorePages(OUTPUT_DIR, ensureDir) {
  // About
  ensureDir(path.join(OUTPUT_DIR, 'about'));
  fs.writeFileSync(path.join(OUTPUT_DIR, 'about', 'index.html'), `${getHeader()}
  <title>About Rapid Service Connect – Your Trusted Emergency Services Referral Network</title>
  <meta name="description" content="Learn about Rapid Service Connect, a free referral service connecting people with trusted emergency service professionals 24/7 across the United States.">
  <link rel="canonical" href="http://rapidserviceconnect.com//about/">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
  <link rel="stylesheet" href="/css/mega-menu.css">
  <link rel="stylesheet" href="/css/chatbot.css">
</head>
<body>
  ${getNavigation(services, cities)}
  <main id="main-content">
    <div class="page-header"><div class="container"><h1>About Rapid Service Connect</h1></div></div>
    <section class="content-section"><div class="container"><article class="content-wrapper">
      <h2>Who We Are</h2>
      <p>Rapid Service Connect is an independent referral service connecting people with qualified emergency service professionals 24 hours a day, 7 days a week. We serve ${cities.length} major U.S. cities across ${services.length} different service categories.</p>
      
      <h2>What We Do</h2>
      <p>We provide free, instant connections to licensed professionals for emergency situations. Our network includes experts in plumbing, HVAC, electrical, roofing, water damage restoration, and many other critical services.</p>
  <link rel="stylesheet" href="/css/mega-menu.css">
  <link rel="stylesheet" href="/css/chatbot.css">
</head>
<body>
  ${getNavigation(services, cities)}
  <main><div class="page-header"><div class="container"><h1>Contact</h1></div></div>
  <section class="content-section"><div class="container"><article class="content-wrapper">
    <h2>Need Emergency Service?</h2>
    <p>Use our <a href="/">service selector</a> to get immediate help with any emergency.</p>
    <h2>General Inquiries</h2>
    <p>For non-emergency questions, please review our <a href="/about/">About page</a> and <a href="/how-it-works/">How It Works</a>.</p>
  </article></div></section></main>
  ${getFooter(services, cities)}`);

  // Legal pages
  ['disclaimer', 'privacy-policy', 'terms'].forEach(page => {
    ensureDir(path.join(OUTPUT_DIR, page));
    fs.writeFileSync(path.join(OUTPUT_DIR, page, 'index.html'), `${getHeader()}
  <title>${page === 'disclaimer' ? 'Disclaimer' : page === 'privacy-policy' ? 'Privacy Policy' : 'Terms of Use'}</title>
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
  <link rel="stylesheet" href="/css/mega-menu.css">
  <link rel="stylesheet" href="/css/chatbot.css">
</head>
<body>
  ${getNavigation(services, cities)}
  <main><div class="page-header"><div class="container"><h1>${page === 'disclaimer' ? 'Disclaimer' : page === 'privacy-policy' ? 'Privacy Policy' : 'Terms of Use'}</h1></div></div>
  <section class="content-section"><div class="container"><article class="content-wrapper">
    ${page === 'disclaimer' ? getDisclaimer() : `<p>Standard ${page === 'privacy-policy' ? 'privacy policy' : 'terms of use'} content goes here.</p>`}
  </article></div></section></main>
  ${getFooter(services, cities)}`);
  });

  console.log('✓ Generated 6 core/legal pages');
}

/**
 * Wrapper for other pages
 */
function generateOtherPages(OUTPUT_DIR, ensureDir) {
  // generateServiceHubPages(OUTPUT_DIR, ensureDir); // Replaced by premium generator
  generateCityHubPages(OUTPUT_DIR, ensureDir);
  generateBlogPages(OUTPUT_DIR, ensureDir);
  generateCorePages(OUTPUT_DIR, ensureDir);
}

module.exports = {
  generateServiceHubPages,
  generateCityHubPages,
  generateBlogPages,
  generateCorePages,
  generateOtherPages
};
