/**
 * RAPID SERVICE CONNECT - ENTERPRISE PAGE GENERATORS
 * ===================================================
 * 
 * Optimized for 1,525+ pages with:
 * - Batch processing
 * - Memory-efficient generation
 * - Advanced SEO optimization
 * - Entity-driven content
 * - LLM-friendly structure
 */

const fs = require('fs');
const path = require('path');
const {
  generateFAQSchema,
  generateHowToSchema,
  generateLocalBusinessSchema,
  generateBreadcrumbSchema,
  generateOrganizationSchema,
  getTrustBadges,
  getHeader,
  getNavigation,
  getFooter,
  getDisclaimer,
  services,
  cities,
  blogArticles
} = require('./seo-utils');

/**
 * Generate Homepage with entity-focused SEO
 */
function generateHomepage(OUTPUT_DIR, ensureDir) {
  const faqItems = [
    {
      question: "Is this a free service?",
      answer: "Yes, Rapid Service Connect is a free referral service. You only pay the service provider directly for their work. There are no hidden fees from us."
    },
    {
      question: "Are you the actual service provider?",
      answer: "No, we are an independent referral service. We connect you with licensed, independent professionals in your area who provide the actual services."
    },
    {
      question: "How quickly can I get help?",
      answer: "Most customers are connected with a service provider within minutes of calling. Response times depend on your location and service availability, but we prioritize speed for all emergency situations."
    },
    {
      question: "Which cities do you currently serve?",
      answer: `We currently serve ${cities.length} major US cities including ${cities.slice(0, 5).map(c => c.name).join(', ')} and many more. We're expanding to additional cities regularly.`
    },
    {
      question: "Are the service providers licensed and insured?",
      answer: "Yes, all service providers in our network are required to maintain proper licensing and insurance for their respective trades. We verify credentials before adding providers to our network."
    },
    {
      question: "What services do you offer?",
      answer: `We connect you with professionals for ${services.length} different emergency services including plumbing, HVAC, electrical, water damage restoration, roofing, and more.`
    },
    {
      question: "Do you operate 24/7?",
      answer: "Yes, our referral service operates 24 hours a day, 7 days a week, including holidays. We connect you with service providers who offer emergency response any time you need help."
    }
  ];

  const breadcrumbs = [{ name: "Home", url: "/" }];

  const html = `${getHeader()}
  <title>24/7 Emergency Services – Connect With Local Pros Instantly | Rapid Service Connect</title>
  <meta name="description" content="Get immediate help for plumbing, HVAC, electrical, roofing, and ${services.length} emergency services. Connect with verified local professionals 24/7 in ${cities.length} U.S. cities.">
  <meta property="og:title" content="24/7 Emergency Services – Instant Local Help">
  <meta property="og:description" content="Connect with verified emergency service professionals instantly. Available 24/7 in ${cities.length} major U.S. cities.">
  <meta property="og:type" content="website">
  <meta property="og:url" content="http://rapidserviceconnect.com/">
  <link rel="canonical" href="http://rapidserviceconnect.com/">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet"  href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
  <link rel="stylesheet" href="/css/mega-menu.css">
  ${generateFAQSchema(faqItems)}
  ${generateOrganizationSchema()}
  ${generateBreadcrumbSchema(breadcrumbs)}
</head>
<body>
  ${getNavigation(services, cities)}
  
  <main id="main-content">
    <section class="hero">
      <div class="container hero-content">
        <h1>Get Connected to Licensed Emergency Professionals in Under 60 Seconds</h1>
        <p class="hero-subtitle">
          Facing an emergency? We instantly connect you with verified, licensed professionals across the United States for plumbing, HVAC, electrical, and ${services.length - 3} more services—available 24/7 in ${cities.length} major U.S. cities nationwide.
        </p>
        ${getTrustBadges()}
        <div class="hero-buttons">
          <a href="#service-selector" class="btn btn-primary btn-large">Get Help Now</a>
          <a href="/how-it-works/" class="btn btn-secondary btn-large">How It Works</a>
        </div>
      </div>
    </section>
    
    <section class="section" id="service-selector">
      <div class="container">
        <div class="service-selector">
          <h2>Find Emergency Services in Your City</h2>
          <p class="text-center mb-6">Select your emergency service and location for immediate professional connection</p>
          
          <form id="service-selector-form" class="selector-form">
            <div class="form-group">
              <label for="service-select">Select Emergency Service</label>
              <select id="service-select" class="form-select" required>
                <option value="">-- Choose Your Emergency Service --</option>
                ${services.map(s => `<option value="${s.slug}">${s.name}</option>`).join('\n                ')}
              </select>
            </div>
            
            <div class="form-group">
              <label for="city-select">Select Your City</label>
              <select id="city-select" class="form-select" required>
                <option value="">-- Choose Your City --</option>
                ${cities.map(c => `<option value="${c.slug}">${c.name}, ${c.state}</option>`).join('\n                ')}
              </select>
            </div>
            
            <button type="submit" class="btn btn-primary btn-large" style="width: 100%;">Connect With Local Professionals Now</button>
          </form>
        </div>
      </div>
    </section>
    
    <section class="section">
      <div class="container">
        <h2 class="text-center mb-8">Emergency Services Available 24/7</h2>
        <div class="services-grid">
          ${services.map(s => `<a href="/${s.slug}/" class="service-card">
            <span class="service-icon" role="img" aria-label="${s.name} icon">${s.icon}</span>
            <h3>${s.name}</h3>
            <p>${s.description}</p>
          </a>`).join('\n          ')}
        </div>
      </div>
    </section>
    
    <section class="section" style="background: var(--color-gray-100);">
      <div class="container">
        <h2 class="text-center mb-8">Serving ${cities.length} Major U.S. Cities</h2>
        <p class="text-center mb-8">Click your city to connect with local emergency service professionals immediately</p>
        <div class="cities-grid">
          ${cities.map(c => `<a href="/${c.slug}/" class="city-link">${c.name}, ${c.state}</a>`).join('\n          ')}
        </div>
      </div>
    </section>
    
    <section class="section">
      <div class="container">
        <h2 class="text-center mb-8">How Rapid Service Connect Works</h2>
        <div class="steps-grid">
          <div class="step-card">
            <div class="step-number" aria-hidden="true">1</div>
            <h3>Choose Service & Location</h3>
            <p>Select the type of emergency service you need and your city from our service selector above. We cover ${cities.length} major U.S. cities.</p>
          </div>
          <div class="step-card">
            <div class="step-number" aria-hidden="true">2</div>
            <h3>Call the Number Provided</h3>
            <p>Get connected instantly with pre-vetted local professionals. No forms, no waiting—just immediate connection to qualified providers.</p>
          </div>
          <div class="step-card">
            <div class="step-number" aria-hidden="true">3</div>
            <h3>Get Professional Help Fast</h3>
            <p>Speak directly with a qualified, licensed service provider in your area who can respond to your emergency immediately—day or night.</p>
          </div>
        </div>
      </div>
    </section>
    
    <section class="section faq-section" style="background: var(--color-gray-100);">
      <div class="container">
        <div class="content-wrapper">
          <h2 class="text-center mb-8">Frequently Asked Questions</h2>
          
          ${faqItems.map(faq => `<div class="faq-item">
            <button class="faq-question">${faq.question}</button>
            <div class="faq-answer">
              <p>${faq.answer}</p>
            </div>
          </div>`).join('\n          ')}
        </div>
      </div>
    </section>
    
    <section class="section">
      <div class="container">
        <div class="content-wrapper">
          ${getDisclaimer()}
        </div>
      </div>
    </section>
  </main>
  
  ${getFooter(services, cities)}`;

  ensureDir(OUTPUT_DIR);
  fs.writeFileSync(path.join(OUTPUT_DIR, 'index.html'), html);
  console.log('✓ Generated SEO-optimized homepage');
}

// Export the generator
module.exports = {
  generateHomepage
};
