/**
 * Rapid Service Connect - AI Chatbot Widget
 * Handles user interaction, data collection, and lead qualification
 */

document.addEventListener('DOMContentLoaded', function () {
    // Inject chatbot HTML
    const chatbotHTML = `
        <div class="chatbot-widget">
            <div class="chatbot-window" id="chatbot-window">
                <div class="chatbot-header">
                    <div class="chatbot-title">
                        <h3>Service Assistant</h3>
                        <span>Online • 24/7 Support</span>
                    </div>
                    <button class="chatbot-close" id="chatbot-close">×</button>
                </div>
                <div class="chatbot-messages" id="chatbot-messages">
                    <!-- Messages will appear here -->
                </div>
                <div class="chatbot-input-area">
                    <input type="text" class="chatbot-input" id="chatbot-input" placeholder="Type your message...">
                    <button class="chatbot-send" id="chatbot-send">➤</button>
                </div>
            </div>
            <button class="chatbot-toggle" id="chatbot-toggle">💬</button>
        </div>
    `;

    document.body.insertAdjacentHTML('beforeend', chatbotHTML);

    // Elements
    const window = document.getElementById('chatbot-window');
    const toggle = document.getElementById('chatbot-toggle');
    const close = document.getElementById('chatbot-close');
    const messages = document.getElementById('chatbot-messages');
    const input = document.getElementById('chatbot-input');
    const sendBtn = document.getElementById('chatbot-send');

    // State
    let step = 0;
    let userData = {
        name: '',
        phone: '',
        service: '',
        city: '',
        issue: ''
    };

    // Toggle window
    toggle.addEventListener('click', () => {
        window.classList.toggle('active');
        if (window.classList.contains('active') && messages.children.length === 0) {
            startConversation();
        }
    });

    close.addEventListener('click', () => {
        window.classList.remove('active');
    });

    // Send message
    function sendMessage() {
        const text = input.value.trim();
        if (text) {
            addMessage(text, 'user');
            input.value = '';
            processUserInput(text);
        }
    }

    sendBtn.addEventListener('click', sendMessage);
    input.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') sendMessage();
    });

    // Add message to UI
    function addMessage(text, sender) {
        const div = document.createElement('div');
        div.className = `message ${sender}`;
        div.innerHTML = text;
        messages.appendChild(div);
        messages.scrollTop = messages.scrollHeight;
    }

    // Show typing indicator
    function showTyping() {
        const div = document.createElement('div');
        div.className = 'typing-indicator';
        div.id = 'typing';
        div.innerHTML = `
            <div class="typing-dot"></div>
            <div class="typing-dot"></div>
            <div class="typing-dot"></div>
        `;
        messages.appendChild(div);
        messages.scrollTop = messages.scrollHeight;
    }

    function hideTyping() {
        const typing = document.getElementById('typing');
        if (typing) typing.remove();
    }

    // Bot response with delay
    function botReply(text, delay = 1000) {
        showTyping();
        setTimeout(() => {
            hideTyping();
            addMessage(text, 'bot');
        }, delay);
    }

    // Conversation Flow
    function startConversation() {
        botReply("👋 Hi there! Welcome to Rapid Service Connect.");
        setTimeout(() => {
            botReply("I can help you connect with a local expert immediately. First, may I have your **full name**?");
        }, 2000);
        step = 1;
    }

    function processUserInput(text) {
        switch (step) {
            case 1: // Name
                userData.name = text;
                botReply(`Nice to meet you, ${text}. What is your **mobile number** so we can keep you updated?`);
                step = 2;
                break;

            case 2: // Phone
                userData.phone = text;
                botReply("Thanks. What specific **service** do you need help with today? (e.g., Plumbing, HVAC, Towing)");
                step = 3;
                break;

            case 3: // Service
                userData.service = text;
                botReply(`Got it, ${userData.service}. And which **city** are you located in?`);
                step = 4;
                break;

            case 4: // City
                userData.city = text;
                botReply("Please describe your issue in detail. 📝");
                setTimeout(() => {
                    botReply("💡 **Tip:** When you speak to the technician, please talk for at least **2-4 minutes** to fully describe the problem. This ensures they bring the right tools and can reach your location directly without issues.");
                }, 1500);
                step = 5;
                break;

            case 5: // Issue
                userData.issue = text;
                console.log("Lead Data Collected:", userData); // For "training purpose"

                botReply("Thank you for the details. I've found a specialist in your area!");
                setTimeout(() => {
                    botReply(`📞 **Call this number immediately:** <br><br><a href="tel:888-555-0123" style="color: var(--color-primary); font-weight: bold; font-size: 18px;">(888) 555-0123</a>`);
                    botReply("Please remember to provide your exact location and details on the call. Thank you for choosing Rapid Service Connect! Visit us again. 🚀");
                }, 1500);
                step = 6;
                break;

            case 6: // End
                botReply("If you need anything else, feel free to call the number above!");
                break;
        }
    }
});
