// ============================================
// RAPID SERVICE CONNECT - ENTERPRISE BUILD SYSTEM
// ============================================
// Generates 1,525+ SEO-optimized pages
// Built with 2024-2025 SEO best practices

const fs = require('fs');
const path = require('path');

// Import specialized generators
const { generateHomepage } = require('./templates/homepage-generator');
const { generateCityMoneyPages } = require('./templates/city-pages-generator');
const { generateOtherPages } = require('./templates/other-generators');
// const { generatePremiumServicePages } = require('./templates/premium-service-generator'); // Temporarily disabled - file corrupted

// Load data for counting
const services = require('./data/services.json');
const cities = require('./data/cities.json');
const blogArticles = require('./data/blog-articles.json');

// Output directory
const OUTPUT_DIR = path.join(__dirname, 'public');

// Ensure directories exist
function ensureDir(dirPath) {
  if (!fs.existsSync(dirPath)) {
    fs.mkdirSync(dirPath, { recursive: true });
  }
}

console.log('');
console.log('🚀 RAPID SERVICE CONNECT - ENTERPRISE BUILD');
console.log('============================================');
console.log(`Building authority site with ${services.length} services × ${cities.length} cities`);
console.log('');
console.log('📊 Page Count Breakdown:');
console.log(`   - Homepage: 1`);
console.log(`   - Service Hubs: ${services.length}`);
console.log(`   - City Money Pages: ${services.length * cities.length} (${services.length} services × ${cities.length} cities)`);
console.log(`   - City Hub Pages: ${cities.length}`);
console.log(`   - Blog Articles: ${blogArticles.length + 1} (including index)`);
console.log(`   - Core/Legal Pages: 6`);
console.log(`   ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
console.log(`   📈 TOTAL: ${1 + services.length + (services.length * cities.length) + cities.length + (blogArticles.length + 1) + 6} pages`);
console.log('');
console.log('🎯 SEO Features:');
console.log('   ✅ FAQ Schema on all service & city pages');
console.log('   ✅ LocalBusiness Schema on city pages');
console.log('   ✅ HowTo Schema support');
console.log('   ✅ Breadcrumb Schema navigation');
console.log('   ✅ Entity-based content structure');
console.log('   ✅ LLM/AI search optimization');
console.log('   ✅ Advanced internal linking');
console.log('   ✅ MarketCall compliance built-in');
console.log('');
console.log('Starting page generation...');
console.log('');

const startTime = Date.now();

try {
  // Generate all pages
  generateHomepage(OUTPUT_DIR, ensureDir);

  // Use BASIC generator for service hubs (premium generator temporarily disabled due to corruption)
  const { generateServiceHubPages } = require('./templates/other-generators');
  generateServiceHubPages(OUTPUT_DIR, ensureDir);

  generateCityMoneyPages(OUTPUT_DIR, ensureDir);
  generateOtherPages(OUTPUT_DIR, ensureDir); // Now excludes service hubs

  const endTime = Date.now();
  const duration = ((endTime - startTime) / 1000).toFixed(2);

  console.log('');
  console.log('✅ BUILD COMPLETE!');
  console.log('==================');
  console.log(`⏱️  Generation time: ${duration} seconds`);
  console.log(`📂 Output directory: ${OUTPUT_DIR}`);
  console.log(`📊 Total pages: ${1 + services.length + (services.length * cities.length) + cities.length + (blogArticles.length + 1) + 6}`);
  console.log('');
  console.log('🎉 Your enterprise authority site is ready!');
  console.log('');
  console.log('📝 Next Steps:');
  console.log('   1. Replace phone number placeholders with MarketCall tracking numbers');
  console.log('   2. Review sample pages for quality');
  console.log('   3. Generate sitemap.xml for faster indexing');
  console.log('   4. Deploy to production');
  console.log('   4. Deploy to production');
  console.log('');

  // Generate sitemaps
  const { generateSitemap } = require('./templates/sitemap-generator');
  generateSitemap(OUTPUT_DIR);

  const { generateHTMLSitemap } = require('./templates/html-sitemap-generator');
  generateHTMLSitemap(OUTPUT_DIR);

} catch (error) {
  console.error('');
  console.error('❌ BUILD FAILED');
  console.error('===============');
  console.error(error);
  process.exit(1);
}
